!This module solves both flow and advection-dispersion equation in 1D karst conduits.
    
MODULE m1D_CV_IGA_conduit 
use mGlobalData
use BASISFUNCTION
implicit none    
! This module solves 1D diffusive wave (DW) flow equation and 1-D advection-dispersion (ADE) 
! equation for 1-D conduits.
! The first part of this module solves 1D diffusive-wave flow approximation 
! (depending variable: piezometric head) &
! of Saint-Venant equations used for unsteady gradually varied flow in open channels. 
! Pressurized flow is modeled by same equation by modifying capacity term.
! Spatial discretization: Control Volume IsoGeometric Analysis (Cv-IGA); prepared for Fup1 or B2-spline.
! Temporal discretization: Backward Euler.    
! Written by: Luka Malenica, 02/2018; see Malenica et al., Water, 2018.
! The second part of module solves 1D ADE; (depending variable: concentration) 
! in 1-D conduits using Euler-Langrangian framework and control volume grid used for flow solution. 
! Advective step is solved by backward particle method so that for every control volume particles
! are tracked backward from the related Gauss/quadrature points using the current time step. Due to advection, 
! concentration is the same along the trajectory along the current time step. It means that concentration 
! is the same in Gauss/quadrature point (indeed end point) as well as in the starting point of trajectory. 
! Finaly, 
! advective concentration is solved as linear problem of function approximation.
! Dispersive step with time term and sink-source (advective flux exchange 3-D matrix) is solved using
! the Control-Volume IsoGeometric Analysis (CV-IGA) in the same way and grid as for flow solution:   
! Spatial discretization: CV-IGA; prepared for Fup1 or B2-spline.
! Temporal discretization: Backward Euler.    
! Written by: Hrvoje Gotovac and Luka Malenica, 01/2023.
 
PRIVATE 

    integer (kind=4) nx,NTOT1D,NNZ
    integer (kind=4) nSegments,nExternal       !Number of characteristic segments and external of basis function
    integer (kind=4) kmax,kmaxCI1D,iter
    integer (kind=4) IURjcnt ,bandP
    integer (kind=4), allocatable :: indx(:)
    integer (kind=4), allocatable :: CNL(:),CNR(:)
    
    real (kind=8), parameter :: eps_NaN=1.d-8
    real (kind=8) dlx1,dlx2
    real (kind=8) dx
    real (kind=8) TStart,TCurr,TStep,TEnd
    real (kind=8) epsNonLin
    real (kind=8) ImplicitUnderRelax,ExplicitUnderRelax,epsUnderRelax
    real (kind=8) ExpUndRelaxStart
    real (kind=8) pMin,pInit,hInit
    real (kind=8) nManningPerf,nManning
    
    real (kind=8), allocatable :: xVertex(:),xCenter(:),xcp1(:),xcp2(:)
    real (kind=8), allocatable :: BFIV(:,:)   !1D basis function integral value for unit caracteristic interval
    real (kind=8), allocatable :: DELX(:)
    real (kind=8), allocatable :: CC_t0(:),CC_lastIter(:),CC_t1(:)  !Solution coefficients for 1D conduit
    real (kind=8), allocatable :: HydHead(:)
    real (kind=8), allocatable :: ASIMQ(:,:),ABAND(:,:),ASIMQ_PROJECTION(:,:),ABAND_PROJECTION(:,:)
    
    character (len=20) LinSol,problem,writeSoE,CheckResidual,FullIntegration
    character (len=20) BCType(2)
    
    logical ConduitClosed  !InitLog/.true./
    
PUBLIC:: InitializeDW1D,SolveDW1D,Hc1D,Vc1D,ResultsDW1D,UpdateCoeffDW1D,UpdateInputDataDW1D,ModifyBCDataDW1D

    CONTAINS

    
!___________________________________________________________!
!PUBLIC DATA

    subroutine InitializeDW1D

        !Read input data
        call InputData
        !Create input data file
        call CreateCalcInputParamFile
        !Prepare calculation parameters
        call CalculateParameters
        !Prepare basis function integral values
        call CalculateIntegrals
        !Transform of initial condition functions
        call InitialConditionTransform
        !Write results in output files
        !call WriteResults('InitialConditions')
        call WriteResults('PrepareFile')
        !Set initial condition as current solution
        CC_t1=CC_LastIter

!stop


    end subroutine
	
 !!!  Solve DW flow equation in 1-D conduits   
    subroutine SolveDW1D(TimeE)
    
        integer (kind=4) IcntEps
        real (kind=8) TimeE
        real (kind=8) epsCurr
        
        TCurr=TimeCurr
        TEnd=TimeE
        dt1D=TEnd-TCurr !Single time step!
        
        !TIME MARCHING - until end of global time step
        DO WHILE(TCurr.lt.TEnd-eps_small)
                        
            !Calculate solution in time t+dt1D
            TCurr=TCurr+dt1D
            epsCurr=2*epsNonLin
            iter=0
            ExpUndRelaxSTART=ExplicitUnderRelax

            !NONLINEAR ITERATIONS   
            do while(epsCurr.gt.epsNonLin)
                !Read input data - for changing parameters during calaculation
                call ReadCalcInputParameters
                iter=iter+1
                !Calculate conduit solution in current time step
                Qsinkhole=0.d0
                call AssembleConduitEquations
                call SolveSystemOfEquations
                call CorrectHydraulicHead
                !Go to next time step if PDE is linear equation
                if(problem.eq.'linear') then
                    !Remember current solution
                    CC_LastIter=CC_t1
                    epsCurr=0.d0
                    exit 
                endif
                !Check for convergence
                call CheckConvergence(epsCurr,IcntEps)
                !Remember current solution
                CC_LastIter=CC_t1
                !Screen record
                if(iter.eq.1) write(*,'(a,10xa,e20.10)') '1D CONDUIT','Time:',TCurr*TimeUnitConvert
                write(*,'(i6,e20.10,i7,2e20.10)') iter,epsCurr,IcntEps,CoeffEps1D,ValueEps1D
                if(iter.eq.1) write(lunFScreen,'(a,10xa,e20.10)') '1D CONDUIT','Time:',TCurr*TimeUnitConvert
                write(lunFScreen,'(i6,e20.10,i7,2e20.10)') iter,epsCurr,IcntEps,CoeffEps1D,ValueEps1D
                !Exit if maximum number of iterations has been reached
                if(iter.ge.maxiter1D) exit
                call CorrectUnderrelaxtionParameters(epsCurr,iter,'iter')
                !Incremental increase of sinkhole discharge
                if(epsCurr.lt.epsSR1D) then
                    SourceRelaxation1D=min(1.d0,SourceRelaxation1D*cfSR1D)
                    call CreateCalcInputParamFile
                endif
                if(iter.eq.resultsIter1D) call ResultsDW1D
    !hp=CX(xp,0,CC_t1)
    !pp=PressureHead(xp,hp)
    !write(*,'(e20.10)') 
            enddo
            
            
            call CorrectUnderrelaxtionParameters(epsCurr,iter,'final')
            !Convergence criteria for KFM
            epsCurrKFM=max(epsCurr,epsCurrKFM)
            epsCurr1D=epsCurr
            iterDW1D=iter

            !!Set current solution as initial codition for next time step
            !CC_t0=CC_t1
            !Multipe time steps need coordinate CC_t0=CC_t1 with KFM loop
    
        ENDDO

    end subroutine

!!!  Solve dispersion step for ADE in 1-D conduits

    subroutine SolveDispersion1D(TimeE)
    
        integer (kind=4) IcntEpsTransport
        real (kind=8) TimeE
        real (kind=8) epsCurr
        integer (kind=4) ii,ix,ic,icnt,jcnt,SimqKS,jj
        real (kind=8) coeff, delta_CCC
        
        TCurr=TimeCurr
        TEnd=TimeE
        dt1D=TEnd-TCurr !Single time step!


        !Set to zero
        icnt=0
        
        !PREPARE SYSTEM OF EQUATIONS
        !X direction
        do ii=-1,nx
            ix=idnint((xCenter(ii)-0.5d0*dx-dlx1)/(dx))
                    
            !Equation counter
            icnt=icnt+1

            !Calculate non-zero coefficinets
            do ic=iX-1,iX+1

                !Coefficients column possition
                jcnt=ic+(nExternal+1)

                !Add firstly time capacity term (so called mass-matrix)
                coeff=BFIV(indx(ii),ic-iX)*DELX(indx(ii))
                coeff = coeff / dt1D
                                

                !Add dispersive term (except in Neumman boundaries)
                
                if (dabs(indx(ii)).ne.2) then
                coeff = coeff - dispersion_conduit * bfun(norder,xVertex(ic),xcp2(ii),dx,1,dlx1,dlx2) 
                coeff = coeff + dispersion_conduit * bfun(norder,xVertex(ic),xcp1(ii),dx,1,dlx1,dlx2)
                else if (indx(ii).eq.-2.and.BCType_Transport(1).eq.'Dirichlet') then 
                coeff = coeff + dispersion_conduit * bfun(norder,xVertex(ic),xcp1(ii),dx,1,dlx1,dlx2)
                else if (indx(ii).eq. 2.and.BCType_Transport(2).eq.'Dirichlet') then 
                coeff = coeff - dispersion_conduit * bfun(norder,xVertex(ic),xcp2(ii),dx,1,dlx1,dlx2)
                end if

                !Add Dirichlet boundary penalization term in the system matrix
                
                if (indx(ii).eq. 2.and.BCType_Transport(2).eq.'Dirichlet') then
                coeff = coeff + bfun(norder,xVertex(ic),xcp2(ii),dx,0,dlx1,dlx2) 
                end if
                
                if (indx(ii).eq.-2.and.BCType_Transport(1).eq.'Dirichlet') then
                coeff = coeff - bfun(norder,xVertex(ic),xcp1(ii),dx,0,dlx1,dlx2)
                end if 

                ! Multiply all previous terms with wetted area in order to get real flux dimensions
                
                coeff = coeff * Area(xVertex(ii),hc1D(xVertex(ii)),Diameter(xVertex(ii)))

                !Add exchange advective term between 1-D conduit and 3-D matrix if advective flux 
                !goes from conduit to the matrix
                
             !Check for matrix around current CV - for karst flow model
            MatrixEx=.false.
            if(xcp1(ii).lt.x2Matrix.AND.xcp2(ii).gt.x1Matrix) then
                if(yConduit.gt.y1Matrix.AND.yConduit.lt.y2Matrix) then
                    if(ConduitBottom(xcp1(ii)).gt.z1Matrix.AND.ConduitBottom(xcp2(ii)).lt.z2Matrix) MatrixEx=.true.
                endif
            endif               
                
                
                coeff = coeff + ExchangeAdvectiveflux_C_M_BFUN(xcp1(ii),xcp2(ii),MatrixEx) 
                
                
                
                !Put calculated value in the ADE system matrix
                if(LinSol.eq.'simq') then
                    ASIMQ(icnt,jcnt)=coeff
                elseif(LinSol.eq.'band') then
                    ABAND(icnt,jcnt-icnt+bandP)=coeff
                endif

            enddo
                    
            !RHS - right hand side
            
            
            !Influence of advective flux from the matrix to the conduit
            
            CCC_t1(icnt) = ExchangeAdvectiveflux_C_M_Conc(xcp1(ii),xcp2(ii),MatrixEx)
            
            !Influence of time term from the start of the current time step (indeed advective concentration)
            
            call trap_int_1D(Conc_LastStep_Conduit,xcp1(ii),xcp2(ii),kmax,delta_CCC)
            CCC_t1(icnt) = CCC_t1(icnt) + (delta_CCC / dt1D)*Area(xVertex(ii),hc1D(xVertex(ii)),Diameter(xVertex(ii)))
            
            !Influence of Neumann boundary conditions
            
            if (indx(ii).eq.-2.and.BCType_Transport(1).eq.'Neumann') then
            CCC_t1(icnt) = CCC_t1(icnt) + BCValue_Transport(1)*Area(xVertex(ii),hc1D(xVertex(ii)),Diameter(xVertex(ii)))
            end if

            if (indx(ii).eq. 2.and.BCType_Transport(2).eq.'Neumann') then
            CCC_t1(icnt) = CCC_t1(icnt) + BCValue_Transport(2)*Area(xVertex(ii),hc1D(xVertex(ii)),Diameter(xVertex(ii)))
            end if

            
            !Influence of Dirichlet boundary conditions (penalization term)
            
            if (indx(ii).eq.-2.and.BCType_Transport(1).eq.'Dirichlet') then
            CCC_t1(icnt) = CCC_t1(icnt) + BCValue_Transport(1)*Area(xVertex(ii),hc1D(xVertex(ii)),Diameter(xVertex(ii)))
            end if

            if (indx(ii).eq. 2.and.BCType_Transport(2).eq.'Dirichlet') then
            CCC_t1(icnt) = CCC_t1(icnt) + BCValue_Transport(2)*Area(xVertex(ii),hc1D(xVertex(ii)),Diameter(xVertex(ii)))
            end if

                               
            
        enddo
                       

!stop    
        !Remember matrix for later usage
        if(LinSol.eq.'simq') ASIMQ_PROJECTION=ASIMQ
        if(LinSol.eq.'band') ABAND_PROJECTION=ABAND
        
        !Solve system of equations
        if(LinSol.eq.'simq') call Simq(ASIMQ,CCC_t1,NTOT1D,SimqKS)
        if(LinSol.eq.'simq'.AND.SimqKS.eq.1) write(*,*) 'SIMQ singularity.'
        if(LinSol.eq.'simq'.AND.SimqKS.eq.1) read (*,*) 
        
        if(LinSol.eq.'band') call BandSol(NTOT1D,2*bandP-1,ABAND,CCC_t1)

!=================================================================================!
       call CheckConvergence_Transport(epsCurrTransport,IcntEpsTransport)       
        !Set conduit solution from time "t" as initial guess for solution in time "t+dt1D"
        CCC_LastIter=CCC_t1
        
!=================================================================================!
 


    end subroutine


!!!  Solve advection step for ADE in 1-D conduits

    subroutine SolveAdvection1D(TimeE)
    
        integer (kind=4) IcntEps
        real (kind=8) TimeE
        real (kind=8) epsCurr
        integer (kind=4) ii,ix,ic,icnt,jcnt,SimqKS,jj
        real (kind=8) coeff
        
        TCurr=TimeCurr
        TEnd=TimeE
        dt1D=TEnd-TCurr !Single time step!
!!!   Check if this time step satisfies Courantov criterion

        !Set to zero
        icnt=0
        
        !PREPARE SYSTEM OF EQUATIONS
        !X direction
        do ii=-1,nx
            ix=idnint((xCenter(ii)-0.5d0*dx-dlx1)/(dx))
                    
            !Equation counter
            icnt=icnt+1

            !Calculate non-zero coefficients
            do ic=iX-1,iX+1
            


                !Coefficients column possition
                jcnt=ic+(nExternal+1)

                !Calculate conduit coefficient
                coeff=BFIV(indx(ii),ic-iX)*DELX(indx(ii))
                                
                !call trap_int_1Dbfun(bfun,nOrder,xVertex(ic),dx,0,dlx1,dlx2,xcp1(ii),xcp2(ii),kmax,coeff)
                                
                !Put value in to conduit array
                if(LinSol.eq.'simq') then
                    ASIMQ(icnt,jcnt)=coeff
                elseif(LinSol.eq.'band') then
                    ABAND(icnt,jcnt-icnt+bandP)=coeff
                endif

            enddo
                    
            !RHS
            
            call trap_int_1D(PTrack_Conduit,xcp1(ii),xcp2(ii),kmax,CCC_t0(icnt))
            
        enddo
                       

!stop    
        !Remember matrix for later usage
        if(LinSol.eq.'simq') ASIMQ_PROJECTION=ASIMQ
        if(LinSol.eq.'band') ABAND_PROJECTION=ABAND
        
        !Solve system of equation
        if(LinSol.eq.'simq') call Simq(ASIMQ,CCC_t0,NTOT1D,SimqKS)
        if(LinSol.eq.'simq'.AND.SimqKS.eq.1) write(*,*) 'SIMQ singularity.'
        if(LinSol.eq.'simq'.AND.SimqKS.eq.1) read (*,*) 
        
        if(LinSol.eq.'band') call BandSol(NTOT1D,2*bandP-1,ABAND,CCC_t0)

!=================================================================================!
        
        !Set conduit solution from time "t" as initial guess for solution in time "t+dt1D"
        CCC_LastIter=CCC_t0
        
!=================================================================================!
 





    end subroutine



    subroutine UpdateInputDataDW1D

        !Boundary conditions: Dirichlet/Neumann/Outflow_H ; Neumann (discharge value)
        BCType(2)='Neumann'     !Upstream - west
        BCType(1)='Outfall_H'   !Downstream - east

    end subroutine

    subroutine UpdateCoeffDW1D(sc)
    
        character(*) sc

        SELECT CASE(sc)
            
        CASE('TimeStepEnd')
            CC_t0=CC_t1
        CASE('TimeStepRestart')
            CC_LastIter=CC_t0
        END SELECT
    end subroutine

    subroutine ModifyBCDataDW1D(sc)
    !Modification for conduit activation. During matrix steady state conduit is closed, after is opened.
    
        character(*) sc

        SELECT CASE(sc)
            
        CASE('ConduitClosed')
            ConduitClosed=.true.
            !Downstream - east
            BCType(1)='Neumann'   
        CASE('ConduitOpened')
            ConduitClosed=.false.
            !Downstream - east
            if(ConduitPipe.eq.'C3') BCType(1)='Dirichlet'
            if(ConduitPipe.eq.'C2') BCType(1)='Outfall_H'
        END SELECT
        
    end subroutine
       
    subroutine ResultsDW1D
    
        call WriteResults('Solution')
        call WriteResults('LevelGraph')
        call CalculateConduitDischarge
        call ScreenRecord('Discharge')

    end subroutine
     
    real (kind=8) function Hc1D(xp)
    !Conduit head solution.

        real(kind=8) xp
        
        Hc1D=CX(xp,0,CC_LastIter)

    end function
    
    real (kind=8) function Vc1D(xp,mdx)
    !Conduit velocity solution.
    
        integer(kind=4) mdx
        real(kind=8) xp
        
        Vc1D=ConduitVelocity(xp,CC_LastIter)

    end function
     
     real (kind=8) function Conc_1D(xp,mdx)
    !Conduit concentration solution.

        real(kind=8) xp
        integer(kind=4) mdx
        
        Conc_1D=CX(xp,mdx,CCC_LastIter)

    end function 
!___________________________________________________________!
!INPUT DATA

    subroutine InputData
    
        !Number of CVs in each direction
        nx=200  !80*2
        
        !Conduit properties
        nManning=0.009d0 !0.009d0  !Manning coefficient D=0.015 -> 0.008    ;    D=0.047 -> 0.01
        if(ConduitPipe.eq.'C2') nManning=0.015d0  
        if(ConduitPipe.eq.'C2') nManningPerf=0.044d0
        pMin=0.005d0
        if(ConduitPipe.eq.'C3') pInit=1.555d0-zConduit !Diameter(dlx2)*2.d0    !0.7d0    !0.05d0
        if(ConduitPipe.eq.'C2') pInit=1.515d0   !1.5045d0-zConduit
        hInit=0.80
        
        !Boundary conditions: Dirichlet/Neumann/Outfall_H/Outflow_H ; Neumann (discharge value)
        !B.C. are modified in ModifyBCDataDW1D
        BCType(2)='Neumann'     !Upstream - west
        if(ConduitPipe.eq.'C3') BCType(1)='Dirichlet'   !Downstream - east
        if(ConduitPipe.eq.'C2') BCType(1)='Outfall_H'   !Downstream - east 
!!!if(ConduitPipe.eq.'C2') BCType(1)='Dirichlet'   !Downstream - east  !!!***!!!

        !CALCULATION PARAMETERS
        
        !Linear system of equations solver
        LinSol='band'       !simq   band
        !Write system of equations Ax=b into data file. Only if LinSol='simq'
        writeSoE='no'
        !Linear or nonlinear problem
        problem='nonlinear'     !linear/nonlinear
        !Nonlinear solver
        epsNonLin=epsConduit1D
        CheckResidual='no'
        !UnderRelax=1.d0 - No Under-Relaxation - Usefull to correct UR factor after 
        ImplicitUnderRelax=IUR1D
        ExplicitUnderRelax=EUR1D
        !Reduce under-relaxation when close to true solution
        epsUnderRelax=epsUR1D
        !Use average CV results for conduit assembly or perform full integration
        FullIntegration='no'
        !Trapezoid rule integration parameter
        kmax=6 
        kmaxCI1D=7
            
    end subroutine
    
    subroutine CreateCalcInputParamFile

        open(lunF1,file='Input1D.inp')
        write(lunF1,*) 'maxiter1D',maxiter1D
        write(lunF1,*) 'epsNonLin',epsNonLin
        write(lunF1,*) 'ImplicitUnderRelax',ImplicitUnderRelax
        write(lunF1,*) 'ExplicitUnderRelax',ExplicitUnderRelax
        write(lunF1,*) 'epsUnderRelax',epsUnderRelax
        write(lunF1,*) 'SourceRelaxation1D',SourceRelaxation1D
        write(lunF1,*) 'epsSR1D',epsSR1D
        write(lunF1,*) 'cfSR1D',cfSR1D
        write(lunF1,*) 'resultsIter1D',resultsIter1D
        close(lunF1)

    end subroutine
    
    subroutine ReadCalcInputParameters
    
        character(len=40) ch,CSwitch
    
        !Control switch
        open(lunF1,file='CSwitch1D.inp')
        read(lunF1,*) CSwitch
        close(lunF1)
        if(CSwitch.eq.'no') return
        
        write(*,*) 'Change parameters and press enter to continue.'
        write(lunFScreen,*) 'Change parameters and press enter to continue.'
        read(*,*)
        
        !Read parameters values
        open(lunF1,file='Input1D.inp')
        read(lunF1,*) ch,maxiter1D
        read(lunF1,*) ch,epsNonLin
        read(lunF1,*) ch,ImplicitUnderRelax
        read(lunF1,*) ch,ExplicitUnderRelax
        read(lunF1,*) ch,epsUnderRelax
        read(lunF1,*) ch,SourceRelaxation1D
        read(lunF1,*) ch,epsSR1D
        read(lunF1,*) ch,cfSR1D
        read(lunF1,*) ch,resultsIter1D
        close(lunF1)
        
        write(*,*) 'maxiter1D',maxiter1D
        write(*,*) 'epsNonLin',epsNonLin
        write(*,*) 'ImplicitUnderRelax',ImplicitUnderRelax
        write(*,*) 'ExplicitUnderRelax',ExplicitUnderRelax
        write(*,*) 'epsUnderRelax',epsUnderRelax
        write(*,*) 'SourceRelaxation1D',SourceRelaxation1D
        write(*,*) 'epsSR1D',epsSR1D
        write(*,*) 'cfSR1D',cfSR1D
        write(*,*) 'resultsIter1D',resultsIter1D
        
        write(lunFScreen,*) 'maxiter1D',maxiter1D
        write(lunFScreen,*) 'epsNonLin',epsNonLin
        write(lunFScreen,*) 'ImplicitUnderRelax',ImplicitUnderRelax
        write(lunFScreen,*) 'ExplicitUnderRelax',ExplicitUnderRelax
        write(lunFScreen,*) 'epsUnderRelax',epsUnderRelax
        write(lunFScreen,*) 'SourceRelaxation1D',SourceRelaxation1D
        write(lunFScreen,*) 'epsSR1D',epsSR1D
        write(lunFScreen,*) 'cfSR1D',cfSR1D
        write(lunFScreen,*) 'resultsIter1D',resultsIter1D
        
        !Return CSwitch to 'no'
        open(lunF1,file='CSwitch1D.inp')
        write(lunF1,*) 'no'
        close(lunF1)        
        
    
    end subroutine
    
!Initial and boundary conditions:

    real (kind=8) function IC_Conduit(xp)
    !Initial condition for hydraulic head in karst conduit.
    
        real (kind=8) xp
        !
        !IC_Conduit=xp**4-8.d0*xp**3-6.d0*xp**2+8*xp+1.d0
        !IC_Conduit=IC_Conduit/sin(0.3*xp)
        !IC_Conduit=tanh((xp-2.3d0)/0.02)
        !IC_Conduit=ConduitBottom(xp)+pInit
        
        IC_Conduit=ConduitBottom(dlx1)+pInit  !Diameter(dlx2)!+(xp-dlx2)/(dlx1-dlx2)*pInit

    end function


!Function which describes concentration advective function from the last time step

    real (kind=8) function Conc_LastStep_Conduit(xp)
    
    
        real (kind=8) xp
        !
        
        Conc_LastStep_Conduit = Conc_1D(xp,0)

    end function

!Function which describes concentration advective function using the particle backward tracking along 
!the current time step

    real (kind=8) function Ptrack_Conduit(xp)
    
    
        real (kind=8) xp
        !
        xp = xp - Vc1D(xp)*dt1D
        if (xp.lt.dlx1) xp=dlx1
        if (xp.gt.dlx2) xp=dlx2
        
        Ptrack_Conduit = Conc_1D(xp,0)

    end function

    real (kind=8) function BCValue(side)
    !Boundary condition values for Dirichlet and Neumann b.c.
    !Other b.c. like Outflow are calculated elsewere.

        integer (kind=4) side
        real (kind=8) TQ0,TQ1,TQ2,TQ3,Qmax
        
        !TQ0=0.d0
        !TQ1=15.d0*60.d0  !15 min
        !TQ2=3.d0*3600.d0  !3 h
        !TQ3=TQ2+TQ1
        !Qmax=-1.41584233d0
        

        !East-downstream
        if(side.eq.1) then
            BCValue=ConduitBottom(dlx2)+Diameter(dlx2)        !-1.d-12     !ConduitBottom(dlx2)+Diameter(dlx2)
            if(ConduitClosed.eq..true.) BCValue=+1.d-12 !No-flow b.c. (Neumann)
            !BCValue=ConduitBottom(dlx2)+2.d0*HeadUnitConvert
        !West-upstream
        elseif(side.eq.2) then
            BCValue=-1.d-12  !ConduitBottom(dlx1)+Diameter(dlx1)*2.d0  !-1.d-12    !ConduitBottom(dlx1)+Diameter(dlx1)+0.2d0
            !if(TCurr.le.TQ1) BCValue=Qmax*(TCurr)/TQ1
            !if(TCurr.gt.TQ1.AND.TCurr.le.TQ2) BCValue=Qmax
            !if(TCurr.gt.TQ2.AND.TCurr.le.TQ3) BCValue=Qmax-Qmax/TQ1*(TCurr-TQ2)
        endif

    end function

    real (kind=8) function SourceTerm(xp)
    !Source term for conduit equation
    
        real (kind=8) xp
        
        SourceTerm=0.d0
        if(TestCase.eq.'C2') return
        !Use second internal and not last CV because of weak imposition of b.c.
        if(abs(xp-xCenter(0)).lt.eps_small) then
            if(TCurr.le.4800.d0) then
                !A
                if(Tcurr.ge.3600.d0) SourceTerm=5.80d0/DischargeUnitConvert     !5.45
                if(Tcurr.gt.4200.d0) SourceTerm=14.65d0/DischargeUnitConvert     !12.2
            endif
        endif
        


    end function

!Conduit Characteristic Functions:    

    real (kind=8) function ConduitBottom(xp)
    !Diameter for circular coduit or width for rectangular conduit
    
        real (kind=8) xp,zU,zD,S,xSC1,S1
        
        !zU=0.2d0
        zD=zConduit
        S=0.01d0  !0.0005d0      !Blago nagnuto korito - potrebne korekcije za brzotok
        !xSC1=dlx2*0.5d0
        !S1=0.0000d0
        
        ConduitBottom=zD+S*(dlx2-xp)
        !if(xp.lt.xSC1) ConduitBottom=zD+S*(dlx2-xSC1)+S1*(XSC1-xp)
               
    end function 

    real (kind=8) function ConduitSlope(xp)
    !Conduit slope calculated by forward difference.
    
        real (kind=8) xp,x1,x2,z1,z2,eps/1.d-2/
         
        x1=xp
        x2=x1+eps
        z1=ConduitBottom(x1)
        z2=ConduitBottom(x2)
        
        ConduitSlope=(z2-z1)/(x2-x1)
               
    end function 

    real (kind=8) function Diameter(xp)
    !Diameter for circular conduit or width for rectangular conduit
    
        real (kind=8) xp
        
        Diameter=ConduitDiameter!*sin(xp)/10.d0+ConduitDiameter
        !if(xp.gt.0.2d0*dlx2+eps_small.AND.xp.lt.0.4d0*dlx2+eps_small) Diameter=ConduitDiameter/4.d0
        !if(xp.gt.0.6d0*dlx2+eps_small.AND.xp.lt.0.8d0*dlx2+eps_small) Diameter=ConduitDiameter/4.d0
               
    end function 

    real (kind=8) function Width(xp,pp,DD)
    !Width of the coundit on the free surface level elevation.
    
        real (kind=8) xp,pp,DD
        
        if(ConduitCrossSection.eq.'circular') then
            !Free surface flow
            if(pp.lt.DD) then
                Width=2.d0*pp*sqrt(DD/pp-1.d0)
            !Pressurized flow
            else
                Width=1000.d0*9.81d0*Area(xp,pp,DD)*4.5d-10
            endif
        elseif(ConduitCrossSection.eq.'rectangular') then
            Width=ConduitRectangularWidth
            if(pp.ge.ConduitRectangularTop) Width=1000.d0*9.81d0*Area(xp,pp,DD)*4.5d-10
        endif
                     
    end function 

    real (kind=8) function Area(xp,pp,DD)
    !Wetted surface
    
        real (kind=8) xp,pp,DD,rr
    
        if(ConduitCrossSection.eq.'circular') then
            !Free surface flow
            if(pp.lt.DD) then
                rr=DD/2.d0
                Area=rr**2*acos(1.d0-pp/rr)+pp*(pp-rr)*sqrt(2.d0*rr/pp-1.d0)
            !Pressurized flow
            else
                Area=DD**2*pi/4.d0
            endif
        elseif(ConduitCrossSection.eq.'rectangular') then
            Area=ConduitRectangularWidth*min(pp,ConduitRectangularTop)
        endif
               
    end function 

    real (kind=8) function Perimeter(xp,pp,DD)
    !Wetted perimeter
    
        real (kind=8) xp,pp,DD
                
        if(ConduitCrossSection.eq.'circular') then
            !Free surface flow
            if(pp.lt.DD) then
                Perimeter=DD*acos(1.d0-2.d0*pp/DD)
            !Pressurized flow
            else
                Perimeter=DD*pi
            endif
        elseif(ConduitCrossSection.eq.'rectangular') then
            !Free surface flow
            if(pp.lt.ConduitRectangularTop) then
                Perimeter=2.d0*pp+ConduitRectangularWidth
            !Pressurized flow
            else
                Perimeter=2.d0*(ConduitRectangularWidth+ConduitRectangularTop)
            endif
        endif
               
    end function 

    real (kind=8) function HydraulicRadius(xp,pp,DD)
    !Wetted perimeter
    
        real (kind=8) xp,pp,DD
        
        HydraulicRadius=Area(xp,pp,DD)/Perimeter(xp,pp,DD)
        
               
    end function 

    real (kind=8) function PressureHead(xp,hp)
    
        real (kind=8) xp,hp
        
        PressureHead=hp-ConduitBottom(xp)
        
        !PressureHead=max(PressureHead,pMin)
        !if(PressureHead.le.0.d0) then
        !    write(*,*) 
        !    write(*,*) 'Negative pressure at xp=',xp,PressureHead
        !    write(*,*) 'Abort.'
        !    stop
        !endif
        
        
               
    end function 

    real (kind=8) function NormalDepth(xp,Qp,ierr)
    !Calculate normal depth for given cross-section and discharge. 
    !Bisection method for nonlinear equation Qp-QQ=0.
    !For pressurized flow function returns top of the conduit.
    
        integer (kind=4) iter,iterMax/100/
        real (kind=8) xp,Qp
        real (kind=8) DD,y0,y1,YN,QQ,S0,eps/1.d-6/
        logical ierr
        
        ierr=.false.
        DD=Diameter(xp)
        S0=ConduitSlope(xp)

        !Initial values
        y0=1.d-4
        if(ConduitCrossSection.eq.'circular')    y1=ConduitDiameter
        if(ConduitCrossSection.eq.'rectangular') y1=ConduitRectangularTop
        
        !Check if flow is pressurized
        YN=y1
        QQ=-1.d0/ManningCoeff(xp)*(HydraulicRadius(xp,YN,DD))**(2.d0/3.d0)*1.d0/sqrt(abs(S0))*S0*Area(xp,YN,DD)
        if(QQ.lt.Qp) then
            NormalDepth=YN
            return
        endif
        
        do iter=0,IterMax
            YN=0.5d0*(y0+y1)
            QQ=-1.d0/ManningCoeff(xp)*(HydraulicRadius(xp,YN,DD))**(2.d0/3.d0)*1.d0/sqrt(abs(S0))*S0*Area(xp,YN,DD)
            if(abs(QQ-Qp).lt.eps) exit
            if(QQ.lt.Qp) y0=YN
            if(QQ.gt.Qp) y1=YN
        enddo
        if(iter.ge.IterMax) ierr=.true.
        NormalDepth=YN
        !if(YN.lt.pMin)
       
    end function 

    real (kind=8) function CriticalDepth(xp,Qp,ierr)
    !Calculate critical depth for given cross-section and discharge. 
    !Bisection method for nonlinear equation dHsdy=1-Fr=0 .
    !For pressurized flow function returns top of the conduit.
    
        integer (kind=4) iter,iterMax/100/
        real (kind=8) xp,Qp
        real (kind=8) DD,yS,yM,YC,Fr,eps/1.d-6/
        real (kind=8) yTop
        logical ierr
        
        ierr=.false.
        DD=Diameter(xp)

        !Initial values
        yS=1.d-4
        yM=100.d0
        if(ConduitCrossSection.eq.'circular')    yTop=ConduitDiameter
        if(ConduitCrossSection.eq.'rectangular') yTop=ConduitRectangularTop

        do iter=0,IterMax
            YC=0.5d0*(yS+yM)
            Fr=FroudNumber(xp,YC,Qp,DD)
            if(abs(1.d0-Fr).lt.eps) exit
            if(Fr.gt.1.d0) yS=YC
            if(Fr.lt.1.d0) yM=YC
            !Finised when pressurized flow
            if(Fr.lt.eps.AND.abs(yC-yTop).lt.eps) then
                yC=yTop
                exit
            endif
        enddo
        if(iter.ge.IterMax) ierr=.true.
        CriticalDepth=YC
       
    end function 

    real (kind=8) function FroudNumber(xp,yp,Qp,DD)
    !Calculates Froud number for given depth and discharge
    
        real (kind=8) xp,yp,Qp,DD
        
        FroudNumber=Width(xp,yp,DD)*Qp**2/(gravity*Area(xp,yp,DD)**3)
        
    end function 

    real (kind=8) function ManningCoeff(xp)
    !Returns Manning coefficinet
    
        real (kind=8) xp
        
        if(ConduitPipe.eq.'C3') then
            ManningCoeff=nManning
        elseif(ConduitPipe.eq.'C2') then
            ManningCoeff=nManning
            !Perforated part
            if(xp.le.xPerforEnd) ManningCoeff=nManningPerf
            !Smooth transition
            if(xp.ge.xPerforEnd.AND.xp.le.xPerforTrans) then
                ManningCoeff=nManning+(nManningPerf-nManning)/(xPerforEnd-xPerforTrans)*(xp-xPerforTrans)
            endif
        endif
        
    end function 
      
!___________________________________________________________!
!PRIVATE SUBROUTINES
        
    subroutine CalculateParameters
    
        integer (kind=4) icnt,ii

        !Number of segments of basis functions
        if(BasisFun.eq.'fup')    nSegments=nOrder+2
        if(BasisFun.eq.'spline') nSegments=nOrder+1
        !Number of external basis functions
        if(BasisFun.eq.'fup')    nExternal=(nOrder+1)/2
        if(BasisFun.eq.'spline') nExternal=(nOrder)/2
        
        !Domain boundaries
        dlx1=x1Conduit   ;   dlx2=x2Conduit

        !Calculate dimensions of CV
        dx=(dlx2-dlx1)/dfloat(nx)

        !Total number of 1D basis functions = total number of CVs
        NTOT1D=(nx+2*nExternal)
 
        !Allocate memory
        allocate(xVertex(-nExternal:nx-1+nExternal),xCenter(-nExternal:nx-1+nExternal))
        allocate(xcp1(-nExternal:nx-1+nExternal),xcp2(-nExternal:nx-1+nExternal))
        allocate(indx(-nExternal:nx-1+nExternal))
        allocate(CC_t0(NTOT1D),CC_LastIter(NTOT1D),CC_t1(NTOT1D))
        allocate(HydHead(NTOT1D))
        allocate(CNL(-2:2),CNR(-2:2),DELX(-2:2))      
        
        !Define finite volume geometry 
        do ii=-nExternal,nx-1+nExternal
            call FVGeometry(ii,nx,dlx1,dlx2,dx,xVertex(ii),xCenter(ii),xcp1(ii),xcp2(ii),indx(ii))
        enddo

        !Calculate total number of non-zero conduit coefficients.
        NNZ=NTOT1D*3**1
        
        !Allocate memory
        if(LinSol.eq.'simq') then
            allocate(ASIMQ(NTOT1D,NTOT1D),ASIMQ_PROJECTION(NTOT1D,NTOT1D))
            ASIMQ=0.d0
        elseif(LinSol.eq.'band') then
            bandP=nSegments
            allocate(ABAND(NTOT1D,2*bandP-1),ABAND_PROJECTION(NTOT1D,2*bandP-1))
            ABAND=0.d0
        endif
        
        !Number of non-zeros basis function left and right in each direction depending of indx
        !Prepared for Fup1 or spline B2; CL(ind)=value
        CNL(-2:2)=(/1,1,1,1,1/)
        CNR(-2:2)=(/1,1,1,1,1/)
        
        !DELXYZ must multiply BFIV to obtain final integral value; DELX(indx)
        DELX(-2:2)=(/dx,dx,dx,dx,dx/)
        
        !Write some information on screen
        call ScreenRecord('CalculateParameters')
            
    end subroutine
    
    subroutine CalculateIntegrals
    !This subroutine prepares 1D integrals of basis function over computational CVs.
    !Calculated values BFIV must be multiplied by dx to obtain real integral value.
    !BFIV(min(ind):max(ind),-nSegments:nSegments)
    !Written for Fup1 or B2-spline
    
        integer (kind=4) icnt,ii,mdx,indx,deriv
        real (kind=8) xv,UnitDX
    
        !Allocate array 
        allocate(BFIV(-nExternal-1:nExternal+1,-nSegments/2:nSegments/2))
        
        !Set to zero
        BFIV=0.d0
        !Calculate for unit UnitDX
        UnitDX=1.d0
        !Calculate only integral of function 0th derivative
        mdx=0
        
        !INTERNAL CV: indx=0; indx=-1; indx=+1
        indx=0
        !All neighbor basis function integrals
        do ii=-nSegments/2,nSegments/2
            xv=dfloat(ii)*UnitDX
            call trap_int_1Dbfun(bfun,norder,xv,UnitDX,mdx,-1.d9,+1.d9,-0.5d0*UnitDX,0.5d0*UnitDX,kmaxCI1D,BFIV(indx,ii))
        enddo
        
        !MODIFIED INTERNAL CV (ii=0, or ii=nx-1): indx=-1 and indx=1
        indx=1
        !All neighbor basis function integrals
        do ii=-nSegments/2,nSegments/2
            xv=dfloat(ii)*UnitDX
            call trap_int_1Dbfun(bfun,norder,xv,UnitDX,mdx,-0.5d0*UnitDX,2.5d0*UnitDX,0.0d0,0.5d0*UnitDX,kmaxCI1D,BFIV(-indx,ii))
            call trap_int_1Dbfun(bfun,norder,xv,UnitDX,mdx,-2.5D0*UnitDX,0.5d0*UnitDX,-0.5d0*UnitDX,0.0d0,kmaxCI1D,BFIV(indx,ii))
        enddo
                
        !BOUNDARY CV (ii=-1, or ii=nx): indx=-2 and indx=2
        indx=2
        !All neighbor basis function values
        do ii=-nSegments/2,nSegments/2
            xv=dfloat(ii)*UnitDX
            call trap_int_1Dbfun(bfun,norder,xv,UnitDX,mdx,-0.5d0*UnitDX,2.5d0*UnitDX,-0.5d0*UnitDX,0.0d0,kmaxCI1D,BFIV(-indx,ii))
            call trap_int_1Dbfun(bfun,norder,xv,UnitDX,mdx,-2.5d0*UnitDX,0.5d0*UnitDX,0.0d0*UnitDX,0.5d0,kmaxCI1D,BFIV(indx,ii))
        enddo
        
        !Partition of unity constant value is not unity for Fup basis functions
        do ii=-nSegments/2,nSegments/2
            xv=dfloat(ii)*UnitDX
            PartOfConstantValue=PartOfConstantValue+bfun(norder,xv,0.d0,UnitDX,mdx,-1.d9,+1.d9)
        enddo
    
    end subroutine
   
    subroutine InitialConditionTransform
    !FINITE VOLUME TRANSFORM OF INITIAL CONDITION.
    !This subroutine approximates initial condition (initial guess for steady-state solution).
    !Results are unknown coefficients at time t=0 needed as initial condition for time integration.

        integer (kind=4) ii,ix,ic,icnt,jcnt,SimqKS,jj
        real (kind=8) coeff

        !Set to zero
        icnt=0
        
        !PREPARE SYSTEM OF EQUATIONS
        !X direction
        do ii=-1,nx
            ix=idnint((xCenter(ii)-0.5d0*dx-dlx1)/(dx))
                    
            !Equation counter
            icnt=icnt+1

            !Calculate non-zero coefficinets
            do ic=iX-1,iX+1

                !Coefficients column possition
                jcnt=ic+(nExternal+1)

                !Calculate conduit coefficient
                coeff=BFIV(indx(ii),ic-iX)*DELX(indx(ii))
                                
                !call trap_int_1Dbfun(bfun,nOrder,xVertex(ic),dx,0,dlx1,dlx2,xcp1(ii),xcp2(ii),kmax,coeff)
                                
                !Put value in to conduit array
                if(LinSol.eq.'simq') then
                    ASIMQ(icnt,jcnt)=coeff
                elseif(LinSol.eq.'band') then
                    ABAND(icnt,jcnt-icnt+bandP)=coeff
                endif

            enddo
                    
            !RHS
            !CC_t0(icnt)=IC_Conduit(xCenter(ii))*(xcp2(ii)-xcp1(ii))
            call trap_int_1D(IC_Conduit,xcp1(ii),xcp2(ii),kmax,CC_t0(icnt))
            
        enddo
                       
        !!Write system of equations into data file
        !if(linsol.eq.'simq'.AND.writeSoE.eq.'yes') then
        !    open(lunF1,file='Ax=b_IC_Conduit.dat')
        !    do ii=1,NTOT1D
        !        do jj=1,NTOT1D    
        !            write(lunF1,'(e30.20,$)') ASIMQ(ii,jj)
        !        enddo
        !        write(lunF1,'(e30.20)') CC_t0(ii)
        !    enddo
        !    close(lunF1)
        !endif
!stop    
        !Remember matrix for later usage
        if(LinSol.eq.'simq') ASIMQ_PROJECTION=ASIMQ
        if(LinSol.eq.'band') ABAND_PROJECTION=ABAND
        
        !Solve system of equation
        if(LinSol.eq.'simq') call Simq(ASIMQ,CC_t0,NTOT1D,SimqKS)
        if(LinSol.eq.'simq'.AND.SimqKS.eq.1) write(*,*) 'SIMQ singularity.'
        if(LinSol.eq.'simq'.AND.SimqKS.eq.1) read (*,*) 
        
        if(LinSol.eq.'band') call BandSol(NTOT1D,2*bandP-1,ABAND,CC_t0)

!=================================================================================!
        
        !Set conduit solution from time "t" as initial guess for solution in time "t+dt1D"
        CC_LastIter=CC_t0
        
!=================================================================================!
 

    end subroutine
     
    subroutine AssembleConduitEquations
    !Assemble system of discretized equations. Coefficinets are calculated by using latest Picard iteration values.
    
        integer (kind=4) ii,ix,ic,icnt,jcnt,SimqKS,side,mdxBC
        real (kind=8) Ke,Kw
        real (kind=8) KDBCe,KDBCw,BFUNe,BFUNw
        real (kind=8) Bij,Kij,Wi,coeff,sumKij
        logical MatrixEx

        !Set to zero
        icnt=0
        CC_t1=0.d0
        if(LinSol.eq.'simq') ASIMQ=0.d0
        if(LinSol.eq.'band') ABAND=0.d0
        
        !PREPARE SYSTEM OF EQUATIONS
        !X direction
        do ii=-1,nx
            ix=idnint((xCenter(ii)-0.5d0*dx-dlx1)/(dx))
                    
            !Equation counter
            icnt=icnt+1
            
            sumKij=0.d0
            
            !Check for matrix around current CV - for karst flow model
            MatrixEx=.false.
            if(xcp1(ii).lt.x2Matrix.AND.xcp2(ii).gt.x1Matrix) then
                if(yConduit.gt.y1Matrix.AND.yConduit.lt.y2Matrix) then
                    if(ConduitBottom(xcp1(ii)).gt.z1Matrix.AND.ConduitBottom(xcp2(ii)).lt.z2Matrix) MatrixEx=.true.
                endif
            endif
            
            !Calculated conveyance factor with latest Picard iteration values
            call ConveyanceMatrixContribution(ii,Ke,Kw)

            !Boundary condition modification
            if(abs(indx(ii)).eq.2) call BoundaryConditionsModifications(ii,Ke,Kw,KDBCe,KDBCw,CC_t1(icnt),side)      

            !Calculate non-zero coefficinets
            do ic=iX-1,iX+1
                
                !Set to zero
                Bij=0.d0
                Kij=0.d0

                !Coefficients column possition
                jcnt=ic+(nExternal+1)

                !Calculate conduit coefficient
                call trap_int_1Dbfun(B_x_BFUN,nOrder,xVertex(ic),dx,0,dlx1,dlx2,xcp1(ii),xcp2(ii),kmax,Bij)
                Kij=-dt1D*( Ke*bfun(norder,xVertex(ic),xcp2(ii),dx,1,dlx1,dlx2)-Kw*bfun(norder,xVertex(ic),xcp1(ii),dx,1,dlx1,dlx2)    )

                !Possible contribution of Dirichlet b.c.
                if(abs(indx(ii)).eq.2) then
                    BFUNe=bfun(nOrder,xVertex(ic),xcp2(ii),dx,0,dlx1,dlx2)
                    BFUNw=bfun(nOrder,xVertex(ic),xcp1(ii),dx,0,dlx1,dlx2)
                    Kij=Kij+KDBCe*BFUNe+KDBCw*BFUNw
                endif
                
                !!Spatial oposite sign stabilization
                !if(OppositeSignStabilization1D.eq.'yes'.AND.indx(ii).eq.0) then
                 !    if(icnt.ne.jcnt) then
                !        sumKij=sumKij+Kij
                !        if(Kij.gt.0.d0) Kij=0
                !    else
                !        Kij=0.d0
                !    endif
                !endif
                
                !Final aij coefficient value
                coeff=cfSS1D*Bij+Kij
                                                
                !Part of RHS contribution from known jcnt coefficient
                CC_t1(icnt)=CC_t1(icnt)+cfSS1D*CC_t0(icnt)*Bij
                                
                !Put value in to conduit array
                if(LinSol.eq.'simq') then
                    ASIMQ(icnt,jcnt)=coeff
                elseif(LinSol.eq.'band') then
                    ABAND(icnt,jcnt-icnt+bandP)=coeff
                endif

            enddo
            
            !!Spatial oposite sign stabilization - correct diagonal term
            !if(OppositeSignStabilization1D.eq.'yes'.AND.indx(ii).eq.0) then
            !    jcnt=icnt   !Diagonal term
            !    if(LinSol.eq.'simq') then
            !        ASIMQ(icnt,jcnt)=ASIMQ(icnt,jcnt)-sumKij
            !    elseif(LinSol.eq.'band') then
            !        ABAND(icnt,jcnt-icnt+bandP)=ABAND(icnt,jcnt-icnt+bandP)-sumKij
            !    endif
            !endif     
                
            !RHS
            !call trap_int_1D(SourceTerm,xcp1(ii),xcp2(ii),kmax,Wi)
            Wi=SourceTerm(xCenter(ii))
            Qsinkhole=Qsinkhole+Wi  !Remember total sink/source discharge
            CC_t1(icnt)=CC_t1(icnt)+dt1D*Wi+dt1D*ExchangeSourceTerm(xcp1(ii),xcp2(ii),MatrixEx)
            
            !Implicit Under-Relaxation
            if(ImplicitUnderRelax.ne.1.d0) then
                if(linsol.eq.'simq') then
                    CC_t1(icnt)=CC_t1(icnt)+(1.d0-ImplicitUnderRelax)/ImplicitUnderRelax*ASIMQ(icnt,icnt)*CC_LastIter(icnt)
                    ASIMQ(icnt,icnt)=ASIMQ(icnt,icnt)/ImplicitUnderRelax
                elseif(linSol.eq.'band') then
                    CC_t1(icnt)=CC_t1(icnt)+(1.d0-ImplicitUnderRelax)/ImplicitUnderRelax*ABAND(icnt,bandP)*CC_LastIter(icnt)
                    ABAND(icnt,bandP)=ABAND(icnt,bandP)/ImplicitUnderRelax
                endif
            endif 
            

        enddo
        
    end subroutine

    subroutine SolveSystemOfEquations
    !Solve discretized system of equations

        integer (kind=4) ii,jj,SimqKS,icnt
        !real (kind=8) cpuTS,cpuTE
        
        !Write system of equations into data file - full direct solver
        if(linsol.eq.'simq'.AND.writeSoE.eq.'yes') then
            open(lunF1,file='Ax=b_DW1D.dat')
            do ii=1,NTOT1D
                do jj=1,NTOT1D    
                    write(lunF1,'(e30.20,$)') ASIMQ(ii,jj)
                enddo
                write(lunF1,'(e30.20)') CC_t1(ii)
            enddo
            close(lunF1)
        endif

!call CPU_TIME(cpuTS)
        !Solve system of equation - full direct solver
        if(LinSol.eq.'simq') call Simq(ASIMQ,CC_t1,NTOT1D,SimqKS)
        if(LinSol.eq.'simq'.AND.SimqKS.eq.1) write(*,*) 'SIMQ singularity.'
        if(LinSol.eq.'simq'.AND.SimqKS.eq.1) read (*,*) 
        
        !Solve system of equation - band direct solver
        if(LinSol.eq.'band') call BandSol(NTOT1D,2*bandP-1,ABAND,CC_t1)
        
!call CPU_TIME(cpuTE)
!write(*,*) 'CPU time (band):',cpuTE-cpuTS
        !Explicit Under-Relaxation
        if(ExplicitUnderRelax.ne.1.d0) then
            do icnt=1,NTOT1D
                CC_t1(icnt)=CC_LastIter(icnt)+ExplicitUnderRelax*(CC_t1(icnt)-CC_LastIter(icnt))
            enddo
        endif
                     
    end subroutine

    subroutine CorrectHydraulicHead
    !Correct solution so there is no negative pressure

        integer (kind=4) ii,icnt,SimqKS
        real (kind=8) xp,zp,pp,hp
        logical correct
        
        correct=.false.
        icnt=0
        do ii=-1,nx
            icnt=icnt+1
            xp=xCenter(ii)
            hp=CX(xp,0,CC_t1) 
            zp=ConduitBottom(xp)
            pp=PressureHead(xp,hp)
            if(pp.lt.pMin) then
                correct=.true.
                pp=pMin
            endif
            HydHead(icnt)=zp+pp
        enddo
        
        !Lower order approximation, i.e. without numerical integration
        if(correct) then
            !Prepare RHS for system
            icnt=0
            do ii=-1,nx
                icnt=icnt+1
                CC_t1(icnt)=HydHead(icnt)*(xcp2(ii)-xcp1(ii))
            enddo
            if(LinSol.eq.'simq') ASIMQ=ASIMQ_PROJECTION 
            if(LinSol.eq.'band') ABAND=ABAND_PROJECTION
            !Solve system 
            if(LinSol.eq.'simq') call Simq(ASIMQ,CC_t1,NTOT1D,SimqKS)
            if(LinSol.eq.'simq'.AND.SimqKS.eq.1) write(*,*) 'SIMQ singularity.'
            if(LinSol.eq.'simq'.AND.SimqKS.eq.1) read (*,*) 
            if(LinSol.eq.'band') call BandSol(NTOT1D,2*bandP-1,ABAND,CC_t1)
        endif
        
             
    end subroutine
       
    subroutine CheckConvergence(eps,icntEps)
    
        integer(kind=4) ii,icnt,icntEps
        real(kind=8) eps,epsTemp
        
        !Skip first iteration
        if(iterKFM.eq.1.AND.iter.eq.1) then
            eps=eps_large
            return
        endif
        
        !Set to zero
        eps=0.d0

        icnt=0
        do ii=-nExternal,nx-1+nExternal
            icnt=icnt+1
            epsTemp=eps
            eps=max(eps,abs(CC_t1(icnt)-CC_lastIter(icnt)))
            if(epsTemp.ne.eps) then
                icntEps=icnt
                CoeffEps1D=CC_t1(icntEps)
                ValueEps1D=CX(xCenter(ii),0,CC_t1)                        
            endif   
        enddo        
        
        if(CheckResidual.eq.'no') return
        !Missing Residual of DE
    
    end subroutine
   
!======================================================!
     subroutine CheckConvergence_Transport(eps,icntEps)
    
        integer(kind=4) ii,icnt,icntEps
        real(kind=8) eps,epsTemp
        
 !       !Skip first iteration
 !       if(iterKFM.eq.1.AND.iter.eq.1) then
 !           eps=eps_large
 !           return
 !       endif
        
        !Set to zero
        eps=0.d0

        icnt=0
        do ii=-nExternal,nx-1+nExternal
            icnt=icnt+1
            epsTemp=eps
            eps=max(eps,abs(CCC_t1(icnt)-CCC_LastIter(icnt)))
            if(epsTemp.ne.eps) then
                icntEps=icnt
 !               CoeffEps1D=CCC_t1(icntEps)
 !               ValueEps1D=CX(xCenter(ii),0,CCC_t1)                        
            endif   
        enddo        
        
 !       if(CheckResidual.eq.'no') return
        !Missing Residual of DE
    
    end subroutine
   
!======================================================!
       
    subroutine CorrectUnderrelaxtionParameters(eps,iter,ch)

        integer (kind=4) iter,I(5)/10,50,400,400,600/
        real (kind=8) eps
        character (len=*) ch
        
        !select case(ch) 
        !case('iter')
        !    if(iter.eq.I(1).AND.eps.gt.1.d0) ExplicitUnderRelax=0.1d0*ExplicitUnderRelax
        !    if(iter.eq.I(2).AND.eps.gt.1.d-2) ExplicitUnderRelax=0.2d0*ExplicitUnderRelax
        !    if(iter.eq.I(3).AND.eps.gt.1.d-3) ExplicitUnderRelax=0.2d0*ExplicitUnderRelax
        !case('final')
        !    ExplicitUnderRelax=ExpUndRelaxSTART
        !end select
        !
        !!Max value = 1.d0
        !ExplicitUnderRelax=min(1.d0,ExplicitUnderRelax)
        
        !!Write into the file
        !call CreateCalcInputParamFile

        !!Reduce under-relaxation when close to true solution
        !if(eps.lt.epsUnderRelax) then
        !    ImplicitUnderRelax=min(1.d0,ImplicitUnderRelax*2.0d0)
        !    ExplicitUnderRelax=min(1.d0,ExplicitUnderRelax*2.0d0)
        !    call CreateCalcInputParamFile
        !endif 
    
    end subroutine        
              
    subroutine FVGeometry(ll,nl,dll1,dll2,dl,lVertex,lCenter,lcp1,lcp2,indl)
    !Define finite volume geometry for specific "l" direction. Fup1 or B2.
    !Input: ll,nl,dll1,dll2,dl
    !Output: lVertex,lCenter,lcp1,lcp2,indl
    implicit none

        integer (kind=4) ll,nl,indl
        real (kind=8) dll1,dll2,dl,lVertex,lCenter,lcp1,lcp2
        
        !Basis function vertex
        lVertex=dll1+dfloat(ll)*dl+dl/2.q0
        !Finite volume centers
        indl=0        
        lCenter=lVertex
        if(ll.eq.-1) then
            indl=-2
            lCenter=dll1+0.25q0*dl
        endif
        if(ll.eq.0) then
            indl=-1
            lCenter=dll1+0.75q0*dl
        endif
        if(ll.eq.nl-1) then
            indl=+1
            lCenter=dll2-0.75q0*dl
        endif
        if(ll.eq.nl) then
            indl=+2
            lCenter=dll2-0.25q0*dl
        endif
        !Finite volume boundaries
        lcp1=lCenter-0.5q0*dl
        lcp2=lCenter+0.5q0*dl
        !Modify first/last two CVs
        if(ll.le.0.OR.ll.ge.nl-1) then
            lcp1=lCenter-0.25q0*dl
            lcp2=lCenter+0.25q0*dl
        endif

    
    end subroutine        
        
    subroutine ConveyanceMatrixContribution(ii,Ke,Kw)
    !Conveyance factor contribution. Calculated with latest Picard iteration values.

        integer (kind=4) ii
        real (kind=8) xe,xw,ze,zw,De,Dw,he,hw,dhedx,dhwdx,pe,pw
        real (kind=8) Ke,Kw,dBFdx_e,dBFdx_w
        
        xe=xcp2(ii)
        xw=xcp1(ii)
        ze=ConduitBottom(xe)
        zw=ConduitBottom(xw)
        De=Diameter(xe)
        Dw=Diameter(xw)
        he=CX(xe,0,CC_LastIter)
        hw=CX(xw,0,CC_LastIter)
        pe=PressureHead(xe,he)
        pw=PressureHead(xw,hw)
        dhedx=abs(CX(xe,1,CC_LastIter))
        dhwdx=abs(CX(xw,1,CC_LastIter))
        dhedx=max(dhedx,eps_NaN)
        dhwdx=max(dhwdx,eps_NaN)
        Ke=1.d0/ManningCoeff(xe)*(HydraulicRadius(xe,pe,De))**(2.d0/3.d0)*Area(xe,pe,De)/sqrt(dhedx)
        Kw=1.d0/ManningCoeff(xw)*(HydraulicRadius(xw,pw,Dw))**(2.d0/3.d0)*Area(xw,pw,Dw)/sqrt(dhwdx)
    
    end subroutine        
                                                    
    subroutine BoundaryConditionsModifications(ii,Ke,Kw,KDBCe,KDBCw,RHS,side)
    !Modifies discretized equation to incorporate boundary conditions.
    
        integer (kind=4) ii,side
        real (kind=8) Ke,Kw,KDBCe,KDBCw,RHS
        
        if(indx(ii).eq.+2) side=1
        if(indx(ii).eq.-2) side=2
                
        !Dirichlet Boundary Condition contribution
        KDBCe=0.d0  
        KDBCw=0.d0
        
        !Weak imposition of b.c.
        if(indx(ii).eq.+2) call WeakBCModif(side,ii,Ke,KDBCe,RHS)
        if(indx(ii).eq.-2) call WeakBCModif(side,ii,Kw,KDBCw,RHS)
     
             
    
    end subroutine        
                                                     
    subroutine WeakBCModif(side,ii,Kf,Kdbcf,RHS)
    !Modifies discretized equation to incorporate boundary conditions weakly.
    
        integer (kind=4) side,ii
        real (kind=8) Kf,Kdbcf,RHS
        real (kind=8) hOut,Qout
        character(len=20) chBC
        !logical DBC,ierr
        
        chBC=BCType(side)          

        !Neumann boundary condition
        if(chBC.eq.'Neumann') then
            Kf=0.d0
            if(side.eq.1) RHS=RHS+dt1D*BCValue(side)
            if(side.eq.2) RHS=RHS-dt1D*BCValue(side)
        endif
           
        !Dirichet boundary condition
        if(chBC.eq.'Dirichlet') then
            Kdbcf=dt1D*BCPenalty1D
            RHS=RHS+dt1D*BCValue(side)*BCPenalty1D
        endif 
                
        !Outfall boundary condition - set as Dirichlet b.c.
        if(chBC.eq.'Outfall_H') then
            Kdbcf=dt1D*BCPenalty1D
            call OutfallBC(side,hOut,Qout)
            RHS=RHS+dt1D*hOut*BCPenalty1D
        endif                    
                            
        !Outflow boundary condition - set as Dirichlet b.c.
        if(chBC.eq.'Outflow_H') then
            Kdbcf=dt1D*BCPenalty1D
            call OutflowBC(side,hOut,Qout)
            RHS=RHS+dt1D*hOut*BCPenalty1D
        endif   
        
        !!Neumann boundary condition
        !if(chBC.eq.'Outflow_Q') then
        !    Kf=0.d0
        !    call OutfallBC(side,hOut,Qout)
        !    if(side.eq.1) RHS=RHS+dt1D*Qout
        !    if(side.eq.2) RHS=RHS-dt1D*Qout
        !endif
             
        
    end subroutine        
                                                
    subroutine OutfallBC(side,hOut,Qout)
    !Calculates outfall b.c.
    
        integer (kind=4) side
        real (kind=8) hOut,Qout
        real (kind=8) DD,xp,hp,pp,Qp,yn,yc,zof
        real (kind=8) xc,pc,hc
        logical ierr
        
        !Outfall boundary condition - set as Dirichlet b.c. by using minimum between normal and critical water depth
        !First calculate discharge on internal face of boundary CV
        if(side.eq.1) xp=xcp1(nx-1)
        if(side.eq.2) xp=xcp2(0)    
        DD=Diameter(xp)
        hp=CX(xp,0,CC_LastIter)
        pp=PressureHead(xp,hp)
        Qp=ConduitVelocity(xp,CC_LastIter)*Area(xp,pp,DD)
        !Find normal and critical depth for calculated discharge
        yN=NormalDepth(xp,Qp,ierr)
        yC=CriticalDepth(xp,Qp,ierr)
        !Use calculated depth for calculate boundary outflow head
        if(side.eq.1) zof=ConduitBottom(dlx2)
        if(side.eq.2) zof=ConduitBottom(dlx1)
        pp=min(yN,yC)
        !Correct head for pressurized (not submerged) flow   
        if(ConduitCrossSection.eq.'rectangular') pp=min(pp,ConduitRectangularTop)
        if(ConduitCrossSection.eq.'circular')    pp=min(pp,DD)
        !if(ConduitCrossSection.eq.'circular'.AND.pp.gt.0.90d0*DD) pp=DD
        hOut=zof+pp        !max(pp,pMin) 
        !!Qout=Qp

    end subroutine        
                                                
    subroutine OutflowBC(side,hOut,Qout)
    !Calculates outflow b.c.
    
        integer (kind=4) side
        real (kind=8) hOut,Qout
        real (kind=8) DD,xp,hp,pp,Qp,yn,yc,zof
        real (kind=8) xc,pc,hc
        logical ierr
 
        !Outflow boundary condition - set as Dirichlet b.c. by using previous CVs center node water depth
        !First calculate discharge on internal face of boundary CV
        if(side.eq.1) xc=xCenter(nx-1)
        if(side.eq.2) xc=xCenter(0)    
        if(side.eq.1) zof=ConduitBottom(dlx2)
        if(side.eq.2) zof=ConduitBottom(dlx1)
        hc=CX(xc,0,CC_LastIter)
        pc=PressureHead(xc,hc)
        hOut=zof+pc     !!!***!!! pp    !max(pp,pMin)
        
        !!Outflow boundary condition - set as Neumann b.c. by using discharge from previous section - need correction in BoundaryConditionsModifications
        !!Calculate discharge on internal face of boundary CV
        !if(side.eq.1) xp=xcp1(nx)    !xcp2(nx-1)
        !if(side.eq.2) xp=xcp2(-1)    !xcp1(0)
        !DD=Diameter(xp)
        !hp=CX(xp,0,CC_LastIter)
        !pp=PressureHead(xp,hp)
        !Qp=ConduitVelocity(xp,CC_LastIter)*Area(xp,pp,DD)
          
    end subroutine        
                                                  
    subroutine trap_int_1D(FF,a,b,kmax,Integ)
    !1D trapezoidal rule integration
    implicit none
        real (kind=8), external:: FF
        real (kind=8) a,b,Integ
        integer (kind=4) kmax
        !
        real (kind=8) hx,xx,sx,f
        integer (kind=8) i,j,icnt
        integer (kind=8) k,nk
        logical (kind=4) ipar

        !Basic step
        hx=(b-a) 
        !Sum of boundary points
        sx=FF(a)+FF(b)
    
        icnt=2
        do k=1,kmax
            nk=2**k
            hx=(b-a)/real(nk)
            do i=0,nk
                if(mod(i,2).eq.0) then
                    ipar=.true.
                else
                    ipar=.false.
                endif

                xx=a+i*hx
                if(ipar) cycle
                f=FF(xx)
                icnt=icnt+1
  
                if(i.eq.0.or.i.eq.nk) then
                    sx=sx+f
                else
                    sx=sx+2*f
                endif
            enddo
        enddo
        
        Integ=sx*hx/2.d0
        
        !write(*,'(a,i12,f20.10)') 'Integral 1D',icnt,Integ
    
    end subroutine trap_int_1D       
                        
    subroutine trap_int_1Dbfun(FF,nord,xv,dx,mdx,X1,X2,a,b,kmax,Integ)
    !1D trapezoidal rule integration
    implicit none
        real (kind=8), external:: FF
        real (kind=8) xv,yv,dx,dy,X1,X2
        real (kind=8) a,b,Integ
        integer (kind=4) nord,mdx,kmax
        !
        real (kind=8) hx,xx,sx,f
        integer (kind=8) i,j,icnt
        integer (kind=8) k,nk
        logical (kind=4) ipar

        !Basic step
        hx=(b-a) 
        !Sum of boundary points
        sx=FF(nord,xv,a,dx,mdx,X1,X2)+FF(nord,xv,b,dx,mdx,X1,X2)
    
        icnt=2
        do k=1,kmax
            nk=2**k
            hx=(b-a)/real(nk)
            do i=0,nk
                if(mod(i,2).eq.0) then
                    ipar=.true.
                else
                    ipar=.false.
                endif

                xx=a+i*hx
                if(ipar) cycle
                f=FF(nord,xv,xx,dx,mdx,X1,X2)
                icnt=icnt+1
  
                if(i.eq.0.or.i.eq.nk) then
                    sx=sx+f
                else
                    sx=sx+2*f
                endif
            enddo
        enddo
        
        Integ=sx*hx/2.d0
        
        !write(*,'(a,i12,f20.10)') 'Integral 1D',icnt,Integ
    
    end subroutine trap_int_1Dbfun       
                
    subroutine Simq(A,B,N,KS)
    !Solves full system of equations by Gauss elimination.

        IMPLICIT real(8) (A-H, O-Z)
        DIMENSION A(1),B(1)
        !
        integer (kind=4) jj,n,ks,j,jy,it,i,ij,imax,i1,k,i2,iqs,&
                         ix,ixj,jx,ixjx,jjx,ny,ia,ib,ic
        
        
    !C
    !C***  FORWARD RJESENJE
    !C
          TOL = 0.000000001q0
          KS = 0
          JJ = -N
          DO 65 J=1,N
          JY=J+1
          JJ=JJ+N+1
          BIGA = 0.0q0
          IT=JJ-J
          DO 30 I=J,N
    !C
    !C***  TRAZANJE MAKSIMALNOG KOEFICIJENTA U STUPCU
    !C
          IJ=IT+I
          IF(abs(BIGA)-abs(A(IJ)))  20,30,30
      20  BIGA = A(IJ)
          IMAX=I
      30  CONTINUE
    !C
    !C***  ISPITIVANJE DA LI JE PIVOT MANJI OD TOLERANCE (SINGULARNA MATRICA)
    !C
          IF(abs(BIGA)-TOL) 35,35,40
      35  KS=1
          RETURN
    !C
    !C***  MEDJUSOBNA ZAMJENA REDAKA AKO JE POTREBNO
    !C
      40  I1=J+N*(J-2)
          IT=IMAX-J
          DO 50 K=J,N
          I1=I1+N
          I2=I1+IT
          SAVE = A(I1)
          A(I1) = A(I2)
          A(I2) = SAVE
    !C
    !C***  DIJELJENJE JEDNADZBE S VODECIM KOEFICIJENTOM
    !C
      50  A(I1) = A(I1)/BIGA
          SAVE = B(IMAX)
          B(IMAX) = B(J)
          B(J) = SAVE/BIGA
    !C
    !C***  ELIMINIRANJE SLIJEDECE VARIJABLE
    !C
          IF(J-N) 55,70,55
      55  IQS=N*(J-1)
          DO 65 IX=JY,N
          IXJ=IQS+IX
          IT=J-IX
          DO 60 JX=JY,N
          IXJX = N*(JX-1)+IX
          JJX=IXJX+IT
      60  A(IXJX) = A(IXJX)-(A(IXJ)*A(JJX))
      65  B(IX) = B(IX)-(B(J)*A(IXJ))
    !C
    !C***  BACK SUPSTITUCIJA
    !C
      70  NY=N-1
          IT=N*N
          DO 80 J=1,NY
          IA=IT-J
          IB=N-J
          IC=N
          DO 80 K=1,J
          B(IB)=B(IB)-A(IA)*B(IC)
          IA=IA-N
      80  IC=IC-1
          RETURN

    end subroutine Simq
      
    subroutine BandSol(nJed,nStup,a,b)
    !Solves system of equations by Gauss elimination in banded form.
        !
        ! Rjesenje nesimetricnog pojasnog sustava linearnih
        ! jednadzbi u stednom obliku matrice Gaussovom eliminacijom
        !
        ! U pozivu:
        !	nJed - broj jednadzbi
        !	nStup - broj diagonala-stupaca u stednom obliku
        !	A - polje koeficijenata matrice u stednom obliku
        !	B - vektor desne strane
        ! U povratku:
        !	B - vektor rjesenja sustava
        !
        !
        implicit none
        integer (kind=4) nJed,nStup
        real(8) a(nJed,nStup),b(nJed)
        real(8) pivot
        integer (kind=4) il,i,j,k,l,m

	        m=(nStup-1)/2+1		!	- pozicija glavne diagonale
        !
	        do k=1,nJed
		        pivot=a(k,m)
		        b(k)=b(k)/pivot
		        il = k+m-1
		        if(il.gt.nJed) il=nJed
		        do j=k,il
			        l=j-k+m
			        a(k,l)=a(k,l)/pivot
		        enddo
		        if(k.eq.nJed) exit
		        do i=k+1,il
			        b(i)=b(i)-a(i,k-i+m)*b(k)
			        do j=k+1,il
				        l=j-i+m
				        a(i,l)=a(i,l)-a(i,k-i+m)*a(k,j-k+m)
			        enddo
		        enddo
	        enddo
        !
	        do k=nJed-1,1,-1
		        il = k+m-1
		        if(il.gt.nJed) il=nJed
		        do j=k+1,il
			        b(k)=b(k)-a(k,j-k+m)*b(j)
		        enddo
	        enddo
    end subroutine BandSol
         
    subroutine ScreenRecord(sc)
    
        character(*) sc        
        
        SELECT CASE(sc)
            
        CASE('CalculateParameters')
            
            write(*,*)
            write(*,*) '1D CONDUIT PARAMETERS'
            write(*,*) 'NTOT1D',NTOT1D
            write(*,*) 'NNZ   ',NNZ
            write(*,'(5(1xa,f12.5,2x))') 'xC1 ',x1Conduit,'xC2 ',x2Conduit,'yC ',yConduit,'zC1 ',ConduitBottom(x1Conduit),'zC2 ',ConduitBottom(x2Conduit)
            write(*,'(1xa,2xf12.5)') 'dx',dx
            write(*,'(1xa,3xe20.10)') 'eps1D',epsConduit1D
            write(*,'(1xa,3xe20.10)') 'IUR1D',IUR1D
            write(*,'(1xa,3xe20.10)') 'EUR1D',EUR1D
            write(*,'(1xa,1xe20.10)') 'epsUR1D',epsUR1D
            write(*,'(1xa,i6)') 'maxiter1D',maxiter1D
            write(*,*) 
            
            write(lunFScreen,*)
            write(lunFScreen,*) '1D CONDUIT PARAMETERS'
            write(lunFScreen,*) 'NTOT1D',NTOT1D
            write(lunFScreen,*) 'NNZ   ',NNZ
            write(lunFScreen,'(5(1xa,f12.5,2x))') 'xC1 ',x1Conduit,'xC2 ',x2Conduit,'yC ',yConduit,'zC1 ',ConduitBottom(x1Conduit),'zC2 ',ConduitBottom(x2Conduit)
            write(lunFScreen,'(1xa,2xf12.5)') 'dx',dx
            write(lunFScreen,'(1xa,3xe20.10)') 'eps1D',epsConduit1D
            write(lunFScreen,'(1xa,3xe20.10)') 'IUR1D',IUR1D
            write(lunFScreen,'(1xa,3xe20.10)') 'EUR1D',EUR1D
            write(lunFScreen,'(1xa,1xe20.10)') 'epsUR1D',epsUR1D
            write(lunFScreen,'(1xa,i6)') 'maxiter1D',maxiter1D
            write(lunFScreen,*) 
            
        CASE('Time')
            write(*,*) 'Current time:',TCurr,'Iterations:',iter
            
            write(lunFScreen,*) 'Current time:',TCurr,'Iterations:',iter
            
        CASE('Discharge')
            write(*,'(4(3xa,f12.8))') 'QCo:',QconduitOutlet*DischargeUnitConvert,'QCi:',QconduitInlet*DischargeUnitConvert,'QCo-QCi:',(QconduitOutlet-QconduitInlet)*DischargeUnitConvert,'Qsinkhole:',Qsinkhole*DischargeUnitConvert
            
            write(lunFScreen,'(4(3xa,f12.8))') 'QCo:',QconduitOutlet*DischargeUnitConvert,'QCi:',QconduitInlet*DischargeUnitConvert,'QCo-QCi:',(QconduitOutlet-QconduitInlet)*DischargeUnitConvert,'Qsinkhole:',Qsinkhole*DischargeUnitConvert
            
        END SELECT

    end subroutine
    
    subroutine WriteResults(sc)
    
        character(*) sc
        character(len=60) zonename 
        integer (kind=4) ii,jj
        real (kind=8) xp,zp,hp,pp,hpOLD,dhdx
        real (kind=8) vx,ic,DD,Qx,Fr,HUC
        real (kind=8) time,P1,P2,Q1,Q2
        
        
        SELECT CASE(sc)
            
        CASE('InitialConditions')
            
            !IC_Conduit approx.
            open(lunF1,file='IC_Conduit_Approx.dat')
            write(lunF1,'(a)') 'TITLE = FFVT_IC_Conduit "'
            write(lunF1,'(a)')  'VARIABLES = "X", "IC", "H" '
            !write(lunF1,'(4(2xa,i6))') 'ZONE I=',nx+2,'F=POINT'
               
            do ii=-1,nx
                xp=xVertex(ii)
                if(xp.lt.dlx1) xp=dlx1
                if(xp.gt.dlx2) xp=dlx2
                ic=IC_Conduit(xp)
                hp=CX(xp,0,CC_t0)
                write(lunF1,'(3e20.10)') xp,ic,hp
            enddo 
            close(lunF1)
        
        CASE('PrepareFile')
            !Conduit solution.
            open(lunF1,file='CONDUIT_RESULTS.dat')
            write(lunF1,'(a)') 'TITLE = FFVM_KFM_IC_4.1"'
            write(lunF1,'(a)')  'VARIABLES = "X", "Z", "D", "P", "H", "dHdx", "Vx", "Qx", "Fr" '
            !write(lunF1,'(4(2xa,i4))') 'ZONE I=',nX+3,'F=POINT'
            close(lunF1)
            
            !Level-graph solution.
            open(lunF1,file='CONDUIT_LevelGraph.dat')
            write(lunF1,'(a)') 'TITLE = FFVM_KFM_IC_4.1"'
            write(lunF1,'(a)')  'VARIABLES = "t", "Pin", "Pout", "Qin", "Qout", "iter" '
            close(lunF1)            

        CASE('Solution')
            
            if(mod(icntTimeStep,nWriteResults).ne.0) return
            
            !Conduit solution.
            open(lunF1,file='CONDUIT_RESULTS.dat',status='old',position='append')
            write(zonename,'(e14.8)') (TimeCurr)*TimeUnitConvert
            zonename=trim(zonename)
            write(lunF1,'(a,a)') 'ZONE T=t_',trim(zonename)

            !Write results in both CV edges and centers
            do ii=-1,nx+1
                do jj=1,2   !jj=1,1 !for CV edges only
                    if(jj.eq.2) then
                        if(ii.eq.nx+1) cycle
                        xp=xCenter(ii)
                    else
                        if(ii.ne.nx+1) xp=xcp1(ii)
                        if(ii.eq.nx+1) xp=xcp2(nx)
                    endif
                    zp=ConduitBottom(xp)
                    DD=Diameter(xp)
                    hp=CX(xp,0,CC_t1)
                    pp=PressureHead(xp,hp)
                    dHdx=CX(xp,1,CC_t1)
                    vx=ConduitVelocity(xp,CC_t1)
                    Qx=vx*Area(xp,pp,DD)
                    Fr=FroudNumber(xp,pp,Qx,DD) 
        !Fr=ManningCoeff(xp)
                    HUC=HeadUnitConvert
                    write(lunF1,'(14e30.20)') xp*HUC,zp*HUC,(zp+DD)*HUC,pp*HUC,hp*HUC,dhdx,vx,Qx*DischargeUnitConvert,Fr
                enddo
            enddo             
            close(lunF1)
            
        CASE('LevelGraph')
            open(lunF1,file='CONDUIT_LevelGraph.dat',status='old',position='append')
            time=(TimeCurr)
            HUC=HeadUnitConvert
            !Inlet
            xp=dlx1
            DD=Diameter(xp)
            hp=CX(xp,0,CC_t1)
            P1=PressureHead(xp,hp)
            Q1=ConduitVelocity(xp,CC_t1)*Area(xp,P1,DD)
            !Outlet
            xp=dlx2
            DD=Diameter(xp)
            hp=CX(xp,0,CC_t1)
            P2=PressureHead(xp,hp) 
            Q2=ConduitVelocity(xp,CC_t1)*Area(xp,P2,DD)
            write(lunF1,'(5e30.20,i5)') time*TimeUnitConvert,P1*HUC,P2*HUC,Q1*DischargeUnitConvert,Q2*DischargeUnitConvert,iter
            close(lunF1)
            
            
        END SELECT
        
    end subroutine
        
    subroutine CalculateConduitDischarge 
    !Conduit discharge through upstream and downstream plane.
    !Discharges are NOT calculated on true boundarys because weak imposition of b.c. &
    !is not truly conservative on edges. Small nonlinear iteration criteria can destroy local conservation property.
    
        integer (kind=4) ii,ix
        real (kind=8) xp,hp,pp,vx,QQ
        
        !Set to zero
        QconduitInlet=0.d0
        QconduitOutlet=0.d0
        
        !Conduit discharge through upstream and downstream plane
        do ii=1,2
            !if(ii.eq.1) xp=dlx1+dx*1.d0    !dlx1
            !if(ii.eq.2) xp=dlx2-dx*1.d0    !dlx2
            !ix=int4((xp-dlx1)/(dx))
            !if(ii.eq.1) xp=xcp1(ix)
            !if(ii.eq.2) xp=xcp2(ix)
            if(ii.eq.1) xp=xcp2(-1)
            if(ii.eq.2) xp=xcp1(nx)
            hp=CX(xp,0,CC_t1)
            pp=PressureHead(xp,hp)
            vx=ConduitVelocity(xp,CC_t1)
            QQ=vx*Area(xp,pp,Diameter(xp))
            if(ii.eq.1) QconduitInlet=QQ
            if(ii.eq.2) QconduitOutlet=QQ
        enddo
        
        !!Calculate conduit in specific location - write in Qsinkhole
        !xp=0.1d0*dlx2
        !hp=CX(xp,0,CC_t1)
        !pp=PressureHead(xp,hp)
        !vx=ConduitVelocity(xp,CC_t1)
        !QconduitXspec=vx*Area(xp,pp,Diameter(xp))
        !Qsinkhole=QconduitXspec
        !

    end subroutine
    
!___________________________________________________________!
!PRIVATE FUNCTIONS
    
    real (kind=8) function B_x_BFUN(norder,xv,xp,dx,mdx,X1,X2)
    !Returns product of conduit capacity term and basis function in xp point. 
    
	    integer (kind=4) norder,mdx
	    real (kind=8) xv,xp,dx,X1,X2
        real (kind=8) hp,pp,DD
    
        hp=CX(xp,0,CC_LastIter)
        pp=PressureHead(xp,hp)
        B_x_BFUN=Width(xp,pp,Diameter(xp))*bfun(norder,xv,xp,dx,mdx,X1,X2)
        
        
    end function 
   
    real(kind=8) function CX(xp,mdx,CC)
    !Returns calculated solution (mdx derivative) value in arbitrary point.
    
        integer(kind=4) mdx
        integer(kind=4) ix,ic,jcnt
        real(kind=8) xp,CC(*)

        CX=0.d0
        !Basis function vertex position
        ix=int4((xp-dlx1)/(dx))
        
        !All non-zero coefficinets
        do ic=iX-1,iX+2
            if(ic.lt.-nExternal.OR.ic.gt.nx-1+nExternal) cycle
            !Ordinal number of basis function
            jcnt=ic+nExternal+1
            !Sum all basis function values multiplied by corresponding coefficients                
            CX=CX+CC(jcnt)*bfun(norder,xVertex(ic),xp,dx,mdx,dlx1,dlx2)             
        enddo

    end function

    real(kind=8) function ConduitVelocity(xp,CC)
    !Returns calculated conduit velocity value in point (xp).
    
        !integer(kind=4) mdx
        real(kind=8) xp,CC(*)
        real(kind=8) hp,pp,dhpdn

        !Head value
        hp=CX(xp,0,CC)
        !Pressure head
        pp=PressureHead(xp,hp)
        !Directional derivative value
        dhpdn=CX(xp,1,CC)
        if(abs(dhpdn).lt.eps_NaN) dhpdn=sign(eps_NaN,dhpdn)
        !Velocity value
        ConduitVelocity=-1.d0/ManningCoeff(xp)*(HydraulicRadius(xp,pp,Diameter(xp)))**(2.d0/3.d0)*1.d0/sqrt(abs(dhpdn))*dhpdn

    end function

!___________________________________________________________!
     
END MODULE
   
